"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StoragesController = void 0;
const abstract_controller_1 = require("../abstract.controller");
const storage_model_1 = require("../../models/storage/storage.model");
const storage_method_1 = require("../../methods/storage/storage.method");
const fs = require("fs");
const path_1 = __importDefault(require("path"));
const format_const_1 = require("../../utils/consts/format.const");
const URL_PUBLIC = process.env.URL_PUBLIC + "/storage";
const MEDIA_PATH = `${__dirname}/../../storage`;
class StoragesController extends abstract_controller_1.AbstractController {
    constructor() {
        super(storage_model_1.StoragesModel);
        this.methods = new storage_method_1.StoragesMethods(storage_model_1.StoragesModel);
    }
    async createItem(req, res) {
        try {
            if (!req.headers.store) {
                this.params.handleErrorResponse(res, "STORE_NOT_EXIST", 411);
                return;
            }
            const fileName = req.headers["x-file-name"];
            if (!fileName) {
                this.params.handleErrorResponse(res, "HEADER_NOT_EXIST", 411);
                return;
            }
            const ext = fileName.split(".").pop();
            const chunkNumber = parseInt(req.headers["x-chunk-number"]);
            const totalChunks = parseInt(req.headers["x-total-chunks"]);
            if (!format_const_1.FORMATS[ext]) {
                this.params.handleErrorResponse(res, "EXTENSION_NOT_AVAILABLE", 411);
                return;
            }
            const fileUrl = `${new Date().getTime()}.${ext}`;
            if (isNaN(chunkNumber) || isNaN(totalChunks) || !fileName) {
                const filePath = path_1.default.join("dist/storage", fileUrl);
                if (!fs.existsSync("dist/storage")) {
                    fs.mkdirSync("dist/storage");
                }
                fs.appendFile(filePath, req.body, async (err) => {
                    if (err) {
                        console.error(err);
                        this.params.handleErrorResponse(res, "ERROR_SAVING_FILE", 500);
                    }
                    const body = {
                        url: `${URL_PUBLIC}/${fileUrl}`,
                        filename: fileName,
                        store_id: req.headers.store,
                        type: format_const_1.FORMATS[ext],
                        ext,
                        category: req.headers.category,
                        user_id: req.user_id,
                    };
                    const response = await storage_model_1.StoragesModel.create(body);
                    res.send(response);
                    return;
                });
            }
            else {
                const appendOptions = { flag: chunkNumber === 0 ? "w" : "a" };
                const filePath = path_1.default.join("dist/storage", fileUrl);
                if (!fs.existsSync("dist/storage")) {
                    fs.mkdirSync("dist/storage");
                }
                fs.appendFile(filePath, req.body, appendOptions, async (err) => {
                    if (err) {
                        console.error(err);
                        this.params.handleErrorResponse(res, "ERROR_SAVING_FILE", 500);
                    }
                    if (chunkNumber === totalChunks - 1 || chunkNumber === totalChunks) {
                        const body = {
                            url: `${URL_PUBLIC}/${fileUrl}`,
                            filename: fileName,
                            store_id: req.headers.store,
                            type: format_const_1.FORMATS[ext],
                            ext,
                            category: req.headers.category,
                            user_id: req.user_id,
                        };
                        const response = await storage_model_1.StoragesModel.create(body);
                        res.send(response);
                        return;
                    }
                    else {
                        res.send({ progress: `${chunkNumber} of ${totalChunks}` });
                        return;
                    }
                });
            }
        }
        catch (e) {
            this.params.handleHttpError(res, e);
        }
    }
    async deleteItem(req, res) {
        try {
            const matchedData = this.params.matchedData(req);
            const _id = matchedData._id;
            const findMedia = await storage_model_1.StoragesModel.findById(_id);
            const fileName = findMedia.filename;
            await storage_model_1.StoragesModel.deleteOne({ _id });
            fs.unlinkSync(`${MEDIA_PATH}/${fileName}`);
            const data = {
                findMedia: fileName,
                deleted: true,
            };
            res.send({ data });
        }
        catch (e) {
            this.params.handleHttpError(res, e);
        }
    }
    async download(req, res) {
        try {
            const matchedData = this.params.matchedData(req);
            const _id = matchedData._id;
            const fileData = await this.model.findById({ _id });
            if (fileData) {
                const filePath = path_1.default.join(__dirname, "../../storage/" + fileData.filename);
                const ext = filePath.split(".").pop();
                const stat = fs.statSync(filePath);
                const fileSize = stat.size;
                const contentType = format_const_1.FORMATS[ext];
                this.methods.sendHlsFile(filePath, fileSize, req, res, contentType);
            }
            else {
                this.params.handleErrorResponse(res, "FILE_NOT_EXIST", 411);
                return;
            }
        }
        catch (e) {
            this.params.handleHttpError(res, e);
        }
    }
}
exports.StoragesController = StoragesController;
