import { StoragesController } from "../../controllers/storage/storage.controller";
import { AuthMiddleware } from "../../middleware/auth.middleware";
import { upload } from "../../utils/handlers/store.handle";
import { StoragesValidator } from "../../validators/storage/storage.validator";
import { AbstractRoute } from "../abstract.route";

const express = require("express");
const router = express.Router();
const controller = new StoragesController();

class StoragesRoute extends AbstractRoute {
  constructor(_controller, _router) {
    super(_controller, _router, new StoragesValidator(), [
      "getItem",
      "getItems",
      "createItem",
      "deleteItem",
      "getDownload"
    ]);
  }
  /**
   * Get all Storage
   * @swagger
   * /storages:
   *    get:
   *      tags:
   *        - Storage
   *      summary: "Get all Storage"
   *      description: "List all Storage with details"
   *      security:
   *        - bearerAuth: []
   *      operationId: "GETstorages"
   *      produces:
   *      - "application/xml"
   *      - "application/json"
   *      parameters:
   *      - $ref: '#/components/parameters/store'
   *      - $ref: '#/components/parameters/pageParam'
   *      - $ref: '#/components/parameters/limitParam'
   *      - name: "user_id"
   *        in: "query"
   *        description: "ID of user of Storage"
   *        required: false
   *      - name: "sort"
   *        in: "query"
   *        description: "The type of sort of items to return."
   *        required: false
   *        schema:
   *          type: "string"
   *          enum: [email, createdAt, -email, -createdAt]
   *      responses:
   *        "200":
   *          description: "successful operation"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/StoragesGetItemsSchema'
   *        "204":
   *          description: "successful operation with empty respose"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/StoragesGetItemsSchema'
   *        "401":
   *          description: "Not allow because you need a Authentication Token"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/ErrorSchema'
   *        '403':
   *          description: "Not allow because you need more permissions"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/ErrorSchema'
   *        "404":
   *          description: "Storage not found"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/ErrorSchema'
   *        default:
   *          description: "Unexpected Error"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/ErrorSchema'
   */
  /**
   * Find Storage by ID
   * @swagger
   * /storages/{id}:
   *    get:
   *      tags:
   *        - Storage
   *      summary: "Find Storage by ID"
   *      description: "Returns a single Storage"
   *      security:
   *        - bearerAuth: []
   *      operationId: "GETstorage"
   *      produces:
   *      - "application/xml"
   *      - "application/json"
   *      parameters:
   *      - $ref: '#/components/parameters/store'
   *      - name: "id"
   *        in: "path"
   *        description: "ID of Storage to return"
   *        required: true
   *      responses:
   *        "200":
   *          description: "successful operation"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/StoragesGetItemSchema'
   *        "204":
   *          description: "successful operation with empty respose"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/StoragesGetItemSchema'
   *        "401":
   *          description: "Not allow because you need a Authentication Token"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/ErrorSchema'
   *        '403':
   *          description: "Not allow because you need more permissions"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/ErrorSchema'
   *        "404":
   *          description: "Storage not found"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/ErrorSchema'
   *        default:
   *          description: "Unexpected Error"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/ErrorSchema'
   */
  /**
   * Add a Storage
   * @swagger
   * /storages:
   *    post:
   *      tags:
   *        - Storage
   *      summary: "Add a Storage"
   *      description: "Add a Storage"
   *      operationId: "ADDstorage"
   *      consumes:
   *        - multipart/form-data
   *      security:
   *        - bearerAuth: []
   *      parameters:
   *      - $ref: '#/components/parameters/store'
   *      - name: "x-chunk-number"
   *        in: "header"
   *        description: "Number of chunks"
   *        required: false
   *      - name: "x-total-chunks"
   *        in: "header"
   *        description: "Total of chunks"
   *        required: false
   *      - name: "x-file-name"
   *        in: "header"
   *        description: "Name of file"
   *        required: true
   *      - name: "category"
   *        in: "header"
   *        description: "Category of file"
   *        required: false
   *      requestBody:
   *        content: 
   *          application/octet-stream:
   *            schema:
   *              type: object
   *              properties:
   *                file:
   *                  type: string
   *                  format: binary
   *      responses:
   *        "200":
   *          description: "successful operation"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/StoragesGetItemSchema'
   *        "201":
   *          description: "Created successfully"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/StoragesGetItemSchema'
   *        "401":
   *          description: "Not allow because you need a Authentication Token"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/ErrorSchema'
   *        '403':
   *          description: "Not allow because you need more permissions"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/ErrorSchema'
   *        "404":
   *          description: "Storage not found"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/ErrorSchema'
   *        default:
   *          description: "Unexpected Error"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/ErrorSchema'
   */
  createItem() {
    this.router.post(
      "/",
      AuthMiddleware,
      // upload.single("file"),
      this.controller.createItem.bind(this.controller)
    );
  }
  /**
   * Delete a Storage
   * @swagger
   * /storages/{id}:
   *    delete:
   *      tags:
   *        - Storage
   *      summary: "Delete a Storage"
   *      description: "Delete an existing Storage"
   *      operationId: "DELETEstorage"
   *      security:
   *        - bearerAuth: []
   *      parameters:
   *      - $ref: '#/components/parameters/store'
   *      - name: "id"
   *        in: "path"
   *        description: "ID of Storage to delete"
   *        required: true
   *      responses:
   *        "200":
   *          description: "successful operation"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/StoragesDeleteItemSchema'
   *        "401":
   *          description: "Not allow because you need a Authentication Token"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/ErrorSchema'
   *        '403':
   *          description: "Not allow because you need more permissions"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/ErrorSchema'
   *        "404":
   *          description: "Storage not found"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/ErrorSchema'
   *        default:
   *          description: "Unexpected Error"
   *          content:
   *            application/json:
   *              schema:
   *                $ref: '#/definitions/ErrorSchema'
   */
  /**
     * Download
     * @swagger
     * /storages/download/{id}:
     *    get:
     *      tags:
     *        - Storage
     *      summary: "Download"
     *      description: "Returns a single Storage"
     *      security:
     *        - bearerAuth: []
     *      operationId: "GETDownloadstorage"
     *      produces:
     *      - "application/xml"
     *      - "application/json"
     *      parameters:
     *      - $ref: '#/components/parameters/store'
     *      - name: "id"
     *        in: "path"
     *        description: "ID of Storage to return"
     *        required: true
     *      responses:
     *        "200":
     *          description: "successful operation"
     *          content:
     *            application/json:
     *              schema:
     *                $ref: '#/definitions/StoragesGetItemSchema'
     *        "204":
     *          description: "successful operation with empty respose"
     *          content:
     *            application/json:
     *              schema:
     *                $ref: '#/definitions/StoragesGetItemSchema'
     *        "401":
     *          description: "Not allow because you need a Authentication Token"
     *          content:
     *            application/json:
     *              schema:
     *                $ref: '#/definitions/ErrorSchema'
     *        '403':
     *          description: "Not allow because you need more permissions"
     *          content:
     *            application/json:
     *              schema:
     *                $ref: '#/definitions/ErrorSchema'
     *        "404":
     *          description: "Storage not found"
     *          content:
     *            application/json:
     *              schema:
     *                $ref: '#/definitions/ErrorSchema'
     *        default:
     *          description: "Unexpected Error"
     *          content:
     *            application/json:
     *              schema:
     *                $ref: '#/definitions/ErrorSchema'
     */
  getDownload() {
    this.router.get(
      "/download/:_id",
      this.validator.getDownload(),
      this.controller.download.bind(this.controller)
    );
  }
}

export const storagesRouter = new StoragesRoute(controller, router).init();
