import { Request, Response } from "express";
import { AbstractController } from "../abstract.controller";
import { StoragesModel } from "../../models/storage/storage.model";
import { StoragesMethods } from "../../methods/storage/storage.method";
const fs = require("fs");
import path from "path";
import { FORMATS } from "../../utils/consts/format.const";

const URL_PUBLIC = process.env.URL_PUBLIC + "/storage";
const MEDIA_PATH = `${__dirname}/../../storage`;

export class StoragesController extends AbstractController {
  methods: StoragesMethods;
  constructor() {
    super(StoragesModel);
    this.methods = new StoragesMethods(StoragesModel);
  }

  /**
   * Upload and create record with public source
   * @param {*} req
   * @param {*} res
   */
  async createItem(req, res: Response) {
    try {
      if (!req.headers.store) {
        this.params.handleErrorResponse(res, "STORE_NOT_EXIST", 411);
        return;
      }
      const fileName = req.headers["x-file-name"];
      if (!fileName) {
        this.params.handleErrorResponse(res, "HEADER_NOT_EXIST", 411);
        return;
      }
      const ext = fileName.split(".").pop();
      const chunkNumber = parseInt(req.headers["x-chunk-number"]);
      const totalChunks = parseInt(req.headers["x-total-chunks"]);
      if (!FORMATS[ext]) {
        this.params.handleErrorResponse(res, "EXTENSION_NOT_AVAILABLE", 411);
        return;
      }
      if (isNaN(chunkNumber) || isNaN(totalChunks) || !fileName) {
        const filePath = path.join("dist/storage", fileName);
        if (!fs.existsSync("dist/storage")) {
          fs.mkdirSync("dist/storage");
        }
        fs.appendFile(filePath, req.body, async (err) => {
          if (err) {
            console.error(err);
            this.params.handleErrorResponse(res, "ERROR_SAVING_FILE", 500);
          }

          const body = {
            url: `${URL_PUBLIC}/${fileName}`,
            filename: fileName,
            store_id: req.headers.store,
            type: FORMATS[ext],
            ext,
            category: req.headers.category,
            user_id: req.user_id,
          };
          const response = await StoragesModel.create(body);
          res.send(response);
          return;
        });
      } else {
        const appendOptions = { flag: chunkNumber === 0 ? "w" : "a" };

        const filePath = path.join("dist/storage", fileName);
        if (!fs.existsSync("dist/storage")) {
          fs.mkdirSync("dist/storage");
        }
        fs.appendFile(filePath, req.body, appendOptions, async (err) => {
          if (err) {
            console.error(err);
            this.params.handleErrorResponse(res, "ERROR_SAVING_FILE", 500);
          }

          if (chunkNumber === totalChunks - 1 || chunkNumber === totalChunks) {
            // Último chunk recibido, la transferencia se ha completado

            const body = {
              url: `${URL_PUBLIC}/${fileName}`,
              filename: fileName,
              store_id: req.headers.store,
              type: FORMATS[ext],
              ext,
              category: req.headers.category,
              user_id: req.user_id,
            };
            const response = await StoragesModel.create(body);
            res.send(response);
            return;
          } else {
            res.send({ progress: `${chunkNumber} of ${totalChunks}` });
            return;
          }
        });
      }
    } catch (e) {
      this.params.handleHttpError(res, e);
    }
  }

  /**
   * delete row
   * @param {*} req
   * @param {*} res
   */
  async deleteItem(req: Request, res: Response) {
    try {
      const matchedData = this.params.matchedData(req);
      const _id = matchedData._id;
      const findMedia = await StoragesModel.findById(_id);
      const fileName = findMedia.filename;
      await StoragesModel.deleteOne({ _id });
      fs.unlinkSync(`${MEDIA_PATH}/${fileName}`);

      const data = {
        findMedia: fileName,
        deleted: true,
      };

      res.send({ data });
    } catch (e) {
      this.params.handleHttpError(res, e);
    }
  }

  /**
   * Download record with public source
   * @param {*} req
   * @param {*} res
   */
  async download(req, res: Response) {
    try {
      const matchedData = this.params.matchedData(req);
      const _id = matchedData._id;
      const fileData = await this.model.findById({ _id });
      if (fileData) {
        const filePath = path.join(
          __dirname,
          "../../storage/" + fileData.filename
        );
        const ext = filePath.split(".").pop();
        const stat = fs.statSync(filePath);
        const fileSize = stat.size;
        const contentType = FORMATS[ext];
        this.methods.sendHlsFile(filePath, fileSize, req, res, contentType);
      } else {
        this.params.handleErrorResponse(res, "FILE_NOT_EXIST", 411);
        return;
      }
    } catch (e) {
      this.params.handleHttpError(res, e);
    }
  }
}
